package explorer;

import cz.cuni.pogamut.Client.Agent;
import cz.cuni.pogamut.Client.RcvMsgEvent;
import cz.cuni.pogamut.Client.RcvMsgListener;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.Triple;
import cz.cuni.pogamut.exceptions.PogamutException;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Logger;

/** Unreal Tournament 2004 Explorer bot.
 * 
 * This is non-violent bot for UT2004. His quest is quite simple - to try for each way-point
 * in the map if it is possible to go directly to other visible way-points. Obtained graph
 * is saved in XML to file named the same as the map. This output file is located in Netbeans root folder.
 * 
 * 
 * XML file structure:
 * 
 * <?xml version="1.0" encoding="ISO-8859-1"?>
 * <UT2004Map Level="DM-Flux2" GameType="BotDeathMatch">
 *      <NavPoint UnrealID="DM-Flux2.InventorySpot95" Location="757.0,1148.0,150.0" >
 *          <NeighbouringNavPoint UnrealID="DM-Flux2.InventorySpot48" LocationX="989.57" LocationY="232.82" LocationZ="-448.0" >
 *              <Jump LocationX="-44.57" LocationY="-202.34" LocationZ="-325.15" />
 *              <Jump LocationX="26.67" LocationY="-20.53" LocationZ="-325.15" />
 *          </NeighbouringNavPoint>
 *          <NeighbouringNavPoint UnrealID="DM-Flux2.PathNode17" Location="-1248.0,-836.0,-271.68" />
 * 	</NavPoint>
 * </UT2004Map>
 * 
 * 
 * @author Jiri Machalek
 * 2008-05-02
 */
public class Main extends Agent {
    
    /** Maximum number of tries for one node. */
    public static int NUMBER_OF_TRIES = 3;
    /** Maximum number of nodes to process.
     * 
     * This is useful for testing, when you don't want to wait too long for example XML file.
     * If you want to process all map, just set it huge enough - something like 1000 for sure.
     */
    public static int NUMBER_OF_ITERATIONS = 1000;
    /** Time limitation for one try in nanoseconds. */
    public static long MAX_NANOSEC_PER_TRY = 5000000000L;
    /** Default orientation for respawn procedure.
     * 
     * I hope it doesn't matter what is set here...
     * Just to hide one awful constant I prefered named it somehow.
     */
    public static Triple DEFAULT_ORIENTATION = new Triple(1,0,0);
    
    /** Currently processed navpoint. */
    protected NavPoint currentNavPoint = null;
    /** Currently processed neighbour for selected navpoint. */
    protected NavPoint currentNeighbour = null;
    /** List of navpoints to process. */
    protected List<NavPoint> allNavPoints = null;
    /** Iterator for list allNavPoints. */
    protected ListIterator<NavPoint> allNPIterator = null;
    /** List of visible neighbours for selected navpoint. */
    protected List<NavPoint> visibleNavPoints = null;
    /** Iterator for list visibleNavPoints. */
    protected ListIterator<NavPoint> visibleNPIterator = null;
    /** List of jump locations. */
    protected List<Triple> jumpLocations = new LinkedList<Triple>();
    /** Counter of tries. */
    protected int tries = 0;
    /** Counter of iterations (see NUMBER_0F_ITERATIONS). */
    protected int iterations = 0;
    /** Counter of navpoints (just for logging). */
    protected int navPointsCounter = 0;
    /** Counter of neighbours (just for logging). */
    protected int neighboursCounter = 0;
    /** Start time for current try. */
    protected long startTime;
    /** Flag for keeping XML structure. */
    protected boolean tagOpened = false;
    /** Output file writer. */
    PrintWriter outFile = null;
    /** Flag whether jump is needed in current try.
     * 
     * This flas is setted in listener for messages WALL_COLLISION.
     */
    protected boolean jumpNeeded = false;
    /** Setting whether to log also unreachable navpoints. */
    protected boolean logUnreachablePoints = true;
    
    /** Creates a new instance of agent. */
    public Main() {
    }

    /** Returns next neighbour to try. */
    protected NavPoint getNextNeighbour() {
        if((visibleNPIterator != null) && visibleNPIterator.hasNext()) {
            return visibleNPIterator.next();
        } else return null;
    }
    
    /** Returns next navpoint to explore. */
    protected NavPoint getNextNavPoint() {
            if(allNPIterator.hasNext() && (iterations < NUMBER_OF_ITERATIONS)) {
                iterations++;
                return allNPIterator.next();
            } else return null;
    }
    
    /** Instance of listener for WALL_COLLISION messages. */
    private WALListener walListener = new WALListener();
    
    /** Listener class for WALL_COLLISION messages. */
    private class WALListener implements RcvMsgListener {

        public WALListener() {
            body.addTypedRcvMsgListener(this, MessageType.WALL_COLLISION);
        }
        @Override
        public void receiveMessage(RcvMsgEvent e) {
            jumpNeeded = true;
        }
        
    }
    
    @Override
    /** Main logic of the bot... the bot itself :-) */
    protected void doLogic() {
        
        /* If currentNavPoint is null then select another navpoint to explore... or die. */
        if(currentNavPoint == null) {
            if(tagOpened) outFile.println("\t</NavPoint>");
            currentNavPoint = getNextNavPoint();
            
            if(currentNavPoint != null) {
                navPointsCounter++;
                neighboursCounter = 0;
                log.info("New point selected (" + navPointsCounter + "/" + allNavPoints.size() + ") - " + currentNavPoint.UnrealID);
                body.respawn(currentNavPoint.location, DEFAULT_ORIENTATION);
                visibleNavPoints = null;
                memory.restartMemory();
                //visibleNavPoints = memory.getSeeNavPoints();
                //visibleNPIterator = visibleNavPoints.listIterator();
                currentNeighbour = null;
                outFile.println("\t<NavPoint UnrealID=\"" + currentNavPoint.UnrealID + "\" LocationX=\"" + currentNavPoint.location.x + "\" LocationY=\"" + currentNavPoint.location.y + "\" LocationZ=\"" + currentNavPoint.location.z + "\" >");
                tagOpened = true;
                return;
                
            } else {
                log.info("Nothing to do...");
                this.stopAgentSoft();
                return;
            }
        }
        
        /* Get new neighbours... Thread has to sleep for a while to let another thread update memory after respawn. */
        if(visibleNavPoints == null) {
            try {
                Thread.sleep(2000);
                visibleNavPoints = memory.getSeeNavPoints();
                visibleNPIterator = visibleNavPoints.listIterator();
            } catch (InterruptedException ex) {
                log.severe("Sleep interrupted!");
            }
        }

        /* If currentNeighbour is null then select another neighbour to explore... or set currentNavPoint to null. */
        if(currentNeighbour == null) {
            currentNeighbour = getNextNeighbour();
            tries = 0;
            jumpLocations.clear();            
            jumpNeeded = false;
            
            if(currentNeighbour != null ) {
                neighboursCounter++;
                log.info("Neighbour selected (" + neighboursCounter + "/" + visibleNavPoints.size() + ") - " + currentNeighbour.UnrealID);
            } else {
                log.info("No more neighbours for " + currentNavPoint.UnrealID);
                currentNavPoint = null;
                return;
            }
        }
        
        /* Now currentNavPoint and currentNeighbour should not be null. */
        if(tries == 0) {
            startTime = System.nanoTime();
            tries++;
        }
        
        /* If time exceeded, start another try or set currentNeighbour to null to select another neighbour. */
        if((System.nanoTime() - startTime) > MAX_NANOSEC_PER_TRY) {
            log.info("Time exceeded - try " + tries);
            if(tries < NUMBER_OF_TRIES) {
                body.respawn(currentNavPoint.location, DEFAULT_ORIENTATION);
                startTime = System.nanoTime();
                jumpLocations.clear();
                tries++;
            } else {
                log.warning("Unreachable neighbour.");
                if(logUnreachablePoints) outFile.println("\t\t<UnreachableNeighbour UnrealID=\"" + currentNeighbour.UnrealID + "\" LocationX=\"" + currentNeighbour.location.x + "\" LocationY=\"" + currentNeighbour.location.y + "\" LocationZ=\"" + currentNeighbour.location.z + "\" />");
                currentNeighbour = null;
                if(visibleNPIterator.hasNext()) body.respawn(currentNavPoint.location, DEFAULT_ORIENTATION);
                return;
            }
        }
        
        /* If bot reached planed neighbour write it with all jumps into the output file and select new neighbour. */
        if(Triple.distanceInSpace(currentNeighbour.location, memory.getAgentLocation()) < 100) {
            log.info("Neighbour reached!");
            if(jumpLocations.isEmpty()) {
                outFile.println("\t\t<NeighbouringNavPoint UnrealID=\"" + currentNeighbour.UnrealID + "\" LocationX=\"" + currentNeighbour.location.x + "\" LocationY=\"" + currentNeighbour.location.y + "\" LocationZ=\"" + currentNeighbour.location.z + "\" />");
            } else {
                outFile.println("\t\t<NeighbouringNavPoint UnrealID=\"" + currentNeighbour.UnrealID + "\" LocationX=\"" + currentNeighbour.location.x + "\" LocationY=\"" + currentNeighbour.location.y + "\" LocationZ=\"" + currentNeighbour.location.z + "\" >");
                for(Triple tr : jumpLocations) {
                    outFile.println("\t\t\t<Jump LocationX=\"" + tr.x + "\" LocationY=\"" + tr.y + "\" LocationZ=\"" + tr.z + "\" />");
                }
                outFile.println("\t\t</NeighbouringNavPoint>");
            }
            currentNeighbour = null;
            if(visibleNPIterator.hasNext()) body.respawn(currentNavPoint.location, DEFAULT_ORIENTATION);
        } else {
        /* Otherwise just try to run to selected neighbour and jump if you need. */
            if(jumpNeeded) {
                body.jump();
                jumpNeeded = false;
                jumpLocations.add(memory.getAgentLocation());
            }
            body.runToNavPoint(currentNeighbour);
        }
    }

    @Override
    protected void prePrepareAgent() throws PogamutException {
    /* Prepares agent logic to run - like initializing neural networks etc.
    not for establishing communication! */
    }

    @Override
    /** Creating output file, writing the headers and initializing of allNavPoints list. */
    protected void postPrepareAgent() throws PogamutException {
        try {
            outFile = new PrintWriter(new FileOutputStream(memory.getGameInfo().level + ".xml"));
        } catch (FileNotFoundException ex) {
            Logger.getLogger(Main.class.getName()).log(Level.SEVERE, null, ex);
        }
        outFile.println("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>");
        outFile.println("<UT2004Map Level=\"" + memory.getGameInfo().level + "\" GameType=\"" + memory.getGameInfo().gameType + "\">");

        allNavPoints = memory.getKnownNavPoints();
        allNPIterator = allNavPoints.listIterator();
    }

    @Override
    /** Closing output file after end of the simulation. */
    protected void shutdownAgent() throws PogamutException {
        outFile.println("</UT2004Map>");
        outFile.close();
    }

    public static void main(String[] args) {
    /*
    DON'T DELETE THIS METHOD, IF YOU DELETE IT NETBEANS WON'T LET YOU RUN THIS 
    BOT. HOWEVER THIS METHOD IS NEVER EXECUTED, THE BOT IS LAUNCHED INSIDE THE 
    NETBEANS BY A CUSTOM ANT TASK (see build.xml).
     */
    }
}
